#!/bin/bash
#
# ------------------------------------------------------
# Client start/stop script for Linux/Unix
# ------------------------------------------------------

#
# CLIENT_INSTALL needs to point to the directory
# where the .jar file is located.
#
CLIENT_INSTALL=/usr/share/LRZ_Sync_Share

#
# CLIENT_HOME sets the location where the client
# will store data like the configuration file,
# internal database, keys, etc. A hidden sub-directory
# with the binary name will be created in that directory.
# PLEASE NOTE: By default this is the user's home
# directory. When you change that, you also need to
# change the symlink to your .config file, which
# resides in your CLIENT_INSTALL directory!
#
CLIENT_HOME=`echo ~`

#
# CLIENT_SILENT defines if the client should start
# in headless mode. If it should start without a
# GUI, set to "true", otherwise leave it at "false".
#
CLIENT_SILENT="false"

# JAVA_MEM specifies some memory settings vor the Java
# virtual machine
#
# If running on a 64-bit system with more than 4GB ram
# or if facing an out-of-memory error the value for the
# -Xmx parameter can be raised (example: -Xmx4g,-Xmx8g,
# -Xmx6g,...), so the virtual machine will be allowed
# to use more memory during runtime.
JAVA_MEM="-Xms64m -Xmx2g -XX:NewRatio=8 -XX:MinHeapFreeRatio=10 -XX:MaxHeapFreeRatio=20"


############################################
### Don't change anything below this line! #
############################################

BINARY_NAME="LRZ_Sync_Share"
DIST_NAME="LRZ Sync+Share"
DIST_VER="$DIST_VER"

# Required since open JDK 17 or newer
JAVA_ADD="--add-exports=java.base/sun.nio.ch=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-opens=java.base/java.lang=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-opens=java.base/java.lang.reflect=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-opens=java.base/java.io=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-exports=jdk.unsupported/sun.misc=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-exports=java.desktop/sun.swing=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-opens=java.desktop/javax.swing.plaf.synth=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-opens=java.desktop/javax.swing.plaf.basic=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-exports=java.desktop/sun.swing.plaf.synth=ALL-UNNAMED"
JAVA_ADD="$JAVA_ADD --add-exports=java.desktop/sun.swing.table=ALL-UNNAMED"

#
# Export variables
#
export CLIENT_INSTALL
export CLIENT_HOME

#
# Check if the user is not root
#
if [ "$(id -u)" = "0" ]; then
	echo "This script must NOT be run as root!" 1>&2
	exit 1
fi

#
# Check .jar file existance
#
if [ ! -f "$CLIENT_INSTALL/LRZ_Sync_Share.jar" ]
then
    echo "LRZ_Sync_Share.jar does not exist in $CLIENT_INSTALL. Please check if $CLIENT_INSTALL is set in the LRZ_Sync_Share-Client.sh file or if LRZ_Sync_Share.jar exists under the specified path!"
    exit 1
fi

#
# Prepare command line for silent launch
#
if [ "$CLIENT_SILENT" = "true" ]; then
	unset DISPLAY
	
	if [  -z "$CLIENT_CONSOLE" ]
	then
		if [ -w /dev/console ]
		then
			CLIENT_CONSOLE=/dev/console
		else
			CLIENT_CONSOLE=/dev/tty
		fi
	fi
	
	if [  -z "$CLIENT_PID" ]
	then
		if [ ! -d "$CLIENT_HOME/.PowerFolder" ]
		then
			mkdir $CLIENT_HOME/.PowerFolder
		fi
		
		CLIENT_PID="$CLIENT_HOME/.PowerFolder/LRZ_Sync_Share.pid"
	fi
fi

# Set classpath
CP="$CLASSPATH:$CLIENT_INSTALL/LRZ_Sync_Share.jar:$CLIENT_INSTALL/libs/*"

#
# Set command line to invoke program start
#
RUN_CMD="$CLIENT_INSTALL/jre/bin/java $JAVA_ADD $JAVA_MEM -Duser.home=$CLIENT_HOME -cp $CP de.dal33t.Start"

#
# Silent mode
#
if [ "$CLIENT_SILENT" = "true" ]; then
	usage()
	{
		echo "Usage: $0 {start|stop|restart|status|console}"
		exit 1
	}
	[ $# -gt 0 ] || usage

	ACTION=$1
	case "$ACTION" in
		start)
			echo "Starting LRZ Sync+Share $DIST_VER"
	
                if [ -f $CLIENT_PID ]
                then
                        if ps -p `cat $CLIENT_PID` >/dev/null ; then
                                echo "Starting LRZ Sync+Share failed: Already Running!"
                                exit 1
                        fi
                fi

			cd $CLIENT_INSTALL
			nohup $RUN_CMD > /dev/null &
			echo $! > $CLIENT_PID
			sleep 2
			echo "STARTED LRZ Sync+Share on: `date`" >> $CLIENT_CONSOLE
			echo "STARTED LRZ Sync+Share with: PID" `cat $CLIENT_PID`
			;;

		console)
			echo "Starting LRZ Sync+Share $DIST_VER"

            if [ -f $CLIENT_PID ]
            then
                if ps -p `cat $CLIENT_PID` >/dev/null ; then
                    echo "Starting LRZ Sync+Share failed: Already Running!"
                    exit 1
                fi
            fi

			echo "STARTED LRZ Sync+Share on: `date`" >> $CLIENT_CONSOLE

			cd $CLIENT_INSTALL
			RUN_CMD="$RUN_CMD -l INFO"
			eval $RUN_CMD
			;;
  
		stop)
			PID=`cat $CLIENT_PID 2>/dev/null`
			eval "$RUN_CMD -k"
			sleep 8
			if [ -n "$PID" ]
			then
				if ps -p $PID > /dev/null
				then
					echo "Stopping LRZ Sync+Share with: PID $PID"
					kill -9 $PID 2>/dev/null
				fi
			fi
			rm -f $CLIENT_PID
			echo "STOPPED LRZ Sync+Share on: `date`" >>$CLIENT_CONSOLE
			;;

		restart)
			$0 stop $*
			sleep 5
			$0 start $*
			;;

		status)
			if [ -f $CLIENT_PID ]
				then
					if ps -p `cat $CLIENT_PID` >/dev/null ; then
						echo "STATUS LRZ Sync+Share: Running PID" `cat $CLIENT_PID`
						exit 0
					else
						echo "STATUS LRZ Sync+Share: Currently not running!"
						exit 1
				fi
			else
				echo "STATUS LRZ Sync+Share: Currently not running!"
				exit 1
			fi
			;;

		*)
			usage
			;;
	esac
	
	exit 0
fi

#
# Non-silent mode
#
eval $RUN_CMD

