/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.websocket.common;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.WritePendingException;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.websocket.api.BatchMode;
import org.eclipse.jetty.websocket.api.RemoteEndpoint;
import org.eclipse.jetty.websocket.api.WriteCallback;
import org.eclipse.jetty.websocket.api.extensions.OutgoingFrames;
import org.eclipse.jetty.websocket.common.BlockingWriteCallback;
import org.eclipse.jetty.websocket.common.LogicalConnection;
import org.eclipse.jetty.websocket.common.WebSocketFrame;
import org.eclipse.jetty.websocket.common.frames.BinaryFrame;
import org.eclipse.jetty.websocket.common.frames.ContinuationFrame;
import org.eclipse.jetty.websocket.common.frames.DataFrame;
import org.eclipse.jetty.websocket.common.frames.PingFrame;
import org.eclipse.jetty.websocket.common.frames.PongFrame;
import org.eclipse.jetty.websocket.common.frames.TextFrame;
import org.eclipse.jetty.websocket.common.io.FrameFlusher;
import org.eclipse.jetty.websocket.common.io.FutureWriteCallback;

public class WebSocketRemoteEndpoint
implements RemoteEndpoint {
    private static final WriteCallback NOOP_CALLBACK = new WriteCallback(){

        @Override
        public void writeSuccess() {
        }

        @Override
        public void writeFailed(Throwable throwable) {
        }
    };
    private static final Logger LOG = Log.getLogger(WebSocketRemoteEndpoint.class);
    private static final int ASYNC_MASK = 65535;
    private static final int BLOCK_MASK = 65536;
    private static final int STREAM_MASK = 131072;
    private static final int PARTIAL_TEXT_MASK = 262144;
    private static final int PARTIAL_BINARY_MASK = 524288;
    private final LogicalConnection connection;
    private final OutgoingFrames outgoing;
    private final AtomicInteger msgState = new AtomicInteger();
    private final BlockingWriteCallback blocker = new BlockingWriteCallback();
    private final AtomicInteger numOutgoingFrames = new AtomicInteger();
    private volatile BatchMode batchMode;
    private int maxNumOutgoingFrames = -1;

    public WebSocketRemoteEndpoint(LogicalConnection logicalConnection, OutgoingFrames outgoingFrames) {
        this(logicalConnection, outgoingFrames, BatchMode.AUTO);
    }

    public WebSocketRemoteEndpoint(LogicalConnection logicalConnection, OutgoingFrames outgoingFrames, BatchMode batchMode) {
        if (logicalConnection == null) {
            throw new IllegalArgumentException("LogicalConnection cannot be null");
        }
        this.connection = logicalConnection;
        this.outgoing = outgoingFrames;
        this.batchMode = batchMode;
    }

    private void blockingWrite(WebSocketFrame webSocketFrame) throws IOException {
        try (BlockingWriteCallback.WriteBlocker writeBlocker = this.blocker.acquireWriteBlocker();){
            this.uncheckedSendFrame(webSocketFrame, writeBlocker);
            writeBlocker.block();
        }
    }

    private boolean lockMsg(MsgType msgType) {
        while (true) {
            int n = this.msgState.get();
            switch (msgType) {
                case BLOCKING: {
                    if ((n & 0xC0000) != 0) {
                        throw new IllegalStateException(String.format("Partial message pending %x for %s", new Object[]{n, msgType}));
                    }
                    if ((n & 0x10000) != 0) {
                        throw new IllegalStateException(String.format("Blocking message pending %x for %s", new Object[]{n, msgType}));
                    }
                    if (!this.msgState.compareAndSet(n, n | 0x10000)) break;
                    return n == 0;
                }
                case ASYNC: {
                    if ((n & 0xC0000) != 0) {
                        throw new IllegalStateException(String.format("Partial message pending %x for %s", new Object[]{n, msgType}));
                    }
                    if ((n & 0xFFFF) == 65535) {
                        throw new IllegalStateException(String.format("Too many async sends: %x", n));
                    }
                    if (!this.msgState.compareAndSet(n, n + 1)) break;
                    return n == 0;
                }
                case STREAMING: {
                    if ((n & 0xC0000) != 0) {
                        throw new IllegalStateException(String.format("Partial message pending %x for %s", new Object[]{n, msgType}));
                    }
                    if ((n & 0x20000) != 0) {
                        throw new IllegalStateException(String.format("Already streaming %x for %s", new Object[]{n, msgType}));
                    }
                    if (!this.msgState.compareAndSet(n, n | 0x20000)) break;
                    return n == 0;
                }
                case PARTIAL_BINARY: {
                    if (n == 524288) {
                        return false;
                    }
                    if (n == 0 && this.msgState.compareAndSet(0, n | 0x80000)) {
                        return true;
                    }
                    throw new IllegalStateException(String.format("Cannot send %s in state %x", new Object[]{msgType, n}));
                }
                case PARTIAL_TEXT: {
                    if (n == 262144) {
                        return false;
                    }
                    if (n == 0 && this.msgState.compareAndSet(0, n | 0x40000)) {
                        return true;
                    }
                    throw new IllegalStateException(String.format("Cannot send %s in state %x", new Object[]{msgType, n}));
                }
            }
        }
    }

    private void unlockMsg(MsgType msgType) {
        while (true) {
            int n = this.msgState.get();
            switch (msgType) {
                case BLOCKING: {
                    if ((n & 0x10000) == 0) {
                        throw new IllegalStateException(String.format("Not Blocking in state %x", n));
                    }
                    if (!this.msgState.compareAndSet(n, n & 0xFFFEFFFF)) break;
                    return;
                }
                case ASYNC: {
                    if ((n & 0xFFFF) == 0) {
                        throw new IllegalStateException(String.format("Not Async in %x", n));
                    }
                    if (!this.msgState.compareAndSet(n, n - 1)) break;
                    return;
                }
                case STREAMING: {
                    if ((n & 0x20000) == 0) {
                        throw new IllegalStateException(String.format("Not Streaming in state %x", n));
                    }
                    if (!this.msgState.compareAndSet(n, n & 0xFFFDFFFF)) break;
                    return;
                }
                case PARTIAL_BINARY: {
                    if (this.msgState.compareAndSet(524288, 0)) {
                        return;
                    }
                    throw new IllegalStateException(String.format("Not Partial Binary in state %x", n));
                }
                case PARTIAL_TEXT: {
                    if (this.msgState.compareAndSet(262144, 0)) {
                        return;
                    }
                    throw new IllegalStateException(String.format("Not Partial Text in state %x", n));
                }
            }
        }
    }

    @Override
    public InetSocketAddress getInetSocketAddress() {
        if (this.connection == null) {
            return null;
        }
        return this.connection.getRemoteAddress();
    }

    private Future<Void> sendAsyncFrame(WebSocketFrame webSocketFrame) {
        FutureWriteCallback futureWriteCallback = new FutureWriteCallback();
        this.uncheckedSendFrame(webSocketFrame, futureWriteCallback);
        return futureWriteCallback;
    }

    @Override
    public void sendBytes(ByteBuffer byteBuffer) throws IOException {
        this.lockMsg(MsgType.BLOCKING);
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug("sendBytes with {}", BufferUtil.toDetailString(byteBuffer));
            }
            this.blockingWrite(new BinaryFrame().setPayload(byteBuffer));
        }
        finally {
            this.unlockMsg(MsgType.BLOCKING);
        }
    }

    @Override
    public Future<Void> sendBytesByFuture(ByteBuffer byteBuffer) {
        this.lockMsg(MsgType.ASYNC);
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug("sendBytesByFuture with {}", BufferUtil.toDetailString(byteBuffer));
            }
            Future<Void> future = this.sendAsyncFrame(new BinaryFrame().setPayload(byteBuffer));
            return future;
        }
        finally {
            this.unlockMsg(MsgType.ASYNC);
        }
    }

    @Override
    public void sendBytes(ByteBuffer byteBuffer, WriteCallback writeCallback) {
        this.lockMsg(MsgType.ASYNC);
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug("sendBytes({}, {})", BufferUtil.toDetailString(byteBuffer), writeCallback);
            }
            this.uncheckedSendFrame(new BinaryFrame().setPayload(byteBuffer), writeCallback == null ? NOOP_CALLBACK : writeCallback);
        }
        finally {
            this.unlockMsg(MsgType.ASYNC);
        }
    }

    public void uncheckedSendFrame(WebSocketFrame webSocketFrame, WriteCallback writeCallback) {
        BatchMode batchMode = BatchMode.OFF;
        if (webSocketFrame.isDataFrame()) {
            batchMode = this.getBatchMode();
        }
        if (this.maxNumOutgoingFrames > 0 && webSocketFrame.isDataFrame()) {
            int n = this.numOutgoingFrames.incrementAndGet();
            writeCallback = WebSocketRemoteEndpoint.from(writeCallback, this.numOutgoingFrames::decrementAndGet);
            if (n > this.maxNumOutgoingFrames) {
                writeCallback.writeFailed(new WritePendingException());
                return;
            }
        }
        this.outgoing.outgoingFrame(webSocketFrame, writeCallback, batchMode);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void sendPartialBytes(ByteBuffer byteBuffer, boolean bl) throws IOException {
        boolean bl2 = this.lockMsg(MsgType.PARTIAL_BINARY);
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug("sendPartialBytes({}, {})", BufferUtil.toDetailString(byteBuffer), bl);
            }
            DataFrame dataFrame = bl2 ? new BinaryFrame() : new ContinuationFrame();
            dataFrame.setPayload(byteBuffer);
            dataFrame.setFin(bl);
            this.blockingWrite(dataFrame);
        }
        finally {
            if (bl) {
                this.unlockMsg(MsgType.PARTIAL_BINARY);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void sendPartialString(String string, boolean bl) throws IOException {
        boolean bl2 = this.lockMsg(MsgType.PARTIAL_TEXT);
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug("sendPartialString({}, {})", string, bl);
            }
            DataFrame dataFrame = bl2 ? new TextFrame() : new ContinuationFrame();
            dataFrame.setPayload(BufferUtil.toBuffer(string, StandardCharsets.UTF_8));
            dataFrame.setFin(bl);
            this.blockingWrite(dataFrame);
        }
        finally {
            if (bl) {
                this.unlockMsg(MsgType.PARTIAL_TEXT);
            }
        }
    }

    @Override
    public void sendPing(ByteBuffer byteBuffer) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("sendPing with {}", BufferUtil.toDetailString(byteBuffer));
        }
        this.sendAsyncFrame(new PingFrame().setPayload(byteBuffer));
    }

    @Override
    public void sendPong(ByteBuffer byteBuffer) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("sendPong with {}", BufferUtil.toDetailString(byteBuffer));
        }
        this.sendAsyncFrame(new PongFrame().setPayload(byteBuffer));
    }

    @Override
    public void sendString(String string) throws IOException {
        this.lockMsg(MsgType.BLOCKING);
        try {
            TextFrame textFrame = new TextFrame().setPayload(string);
            if (LOG.isDebugEnabled()) {
                LOG.debug("sendString with {}", BufferUtil.toDetailString(textFrame.getPayload()));
            }
            this.blockingWrite(textFrame);
        }
        finally {
            this.unlockMsg(MsgType.BLOCKING);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Future<Void> sendStringByFuture(String string) {
        this.lockMsg(MsgType.ASYNC);
        try {
            TextFrame textFrame = new TextFrame().setPayload(string);
            if (LOG.isDebugEnabled()) {
                LOG.debug("sendStringByFuture with {}", BufferUtil.toDetailString(textFrame.getPayload()));
            }
            Future<Void> future = this.sendAsyncFrame(textFrame);
            return future;
        }
        finally {
            this.unlockMsg(MsgType.ASYNC);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void sendString(String string, WriteCallback writeCallback) {
        this.lockMsg(MsgType.ASYNC);
        try {
            TextFrame textFrame = new TextFrame().setPayload(string);
            if (LOG.isDebugEnabled()) {
                LOG.debug("sendString({},{})", BufferUtil.toDetailString(textFrame.getPayload()), writeCallback);
            }
            this.uncheckedSendFrame(textFrame, writeCallback == null ? NOOP_CALLBACK : writeCallback);
        }
        finally {
            this.unlockMsg(MsgType.ASYNC);
        }
    }

    @Override
    public BatchMode getBatchMode() {
        return this.batchMode;
    }

    @Override
    public void setBatchMode(BatchMode batchMode) {
        this.batchMode = batchMode;
    }

    @Override
    public int getMaxOutgoingFrames() {
        return this.maxNumOutgoingFrames;
    }

    @Override
    public void setMaxOutgoingFrames(int n) {
        this.maxNumOutgoingFrames = n;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void flush() throws IOException {
        this.lockMsg(MsgType.ASYNC);
        try (BlockingWriteCallback.WriteBlocker writeBlocker = this.blocker.acquireWriteBlocker();){
            this.uncheckedSendFrame(FrameFlusher.FLUSH_FRAME, writeBlocker);
            writeBlocker.block();
        }
        finally {
            this.unlockMsg(MsgType.ASYNC);
        }
    }

    public String toString() {
        return String.format("%s@%x[batching=%b]", new Object[]{this.getClass().getSimpleName(), this.hashCode(), this.getBatchMode()});
    }

    private static WriteCallback from(final WriteCallback writeCallback, final Runnable runnable) {
        return new WriteCallback(){

            @Override
            public void writeFailed(Throwable throwable) {
                try {
                    writeCallback.writeFailed(throwable);
                }
                finally {
                    runnable.run();
                }
            }

            @Override
            public void writeSuccess() {
                try {
                    writeCallback.writeSuccess();
                }
                finally {
                    runnable.run();
                }
            }
        };
    }

    private static enum MsgType {
        BLOCKING,
        ASYNC,
        STREAMING,
        PARTIAL_TEXT,
        PARTIAL_BINARY;

    }
}

