/*
 * Decompiled with CFR 0.152.
 */
package org.apache.http.impl.client.cache;

import java.util.Date;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.Header;
import org.apache.http.HeaderElement;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.annotation.Contract;
import org.apache.http.annotation.ThreadingBehavior;
import org.apache.http.client.cache.HttpCacheEntry;
import org.apache.http.client.utils.DateUtils;
import org.apache.http.impl.client.cache.CacheConfig;
import org.apache.http.impl.client.cache.CacheValidityPolicy;

@Contract(threading=ThreadingBehavior.IMMUTABLE_CONDITIONAL)
class CachedResponseSuitabilityChecker {
    private final Log log = LogFactory.getLog(this.getClass());
    private final boolean sharedCache;
    private final boolean useHeuristicCaching;
    private final float heuristicCoefficient;
    private final long heuristicDefaultLifetime;
    private final CacheValidityPolicy validityStrategy;

    CachedResponseSuitabilityChecker(CacheValidityPolicy cacheValidityPolicy, CacheConfig cacheConfig) {
        this.validityStrategy = cacheValidityPolicy;
        this.sharedCache = cacheConfig.isSharedCache();
        this.useHeuristicCaching = cacheConfig.isHeuristicCachingEnabled();
        this.heuristicCoefficient = cacheConfig.getHeuristicCoefficient();
        this.heuristicDefaultLifetime = cacheConfig.getHeuristicDefaultLifetime();
    }

    CachedResponseSuitabilityChecker(CacheConfig cacheConfig) {
        this(new CacheValidityPolicy(), cacheConfig);
    }

    private boolean isFreshEnough(HttpCacheEntry httpCacheEntry, HttpRequest httpRequest, Date date) {
        if (this.validityStrategy.isResponseFresh(httpCacheEntry, date)) {
            return true;
        }
        if (this.useHeuristicCaching && this.validityStrategy.isResponseHeuristicallyFresh(httpCacheEntry, date, this.heuristicCoefficient, this.heuristicDefaultLifetime)) {
            return true;
        }
        if (this.originInsistsOnFreshness(httpCacheEntry)) {
            return false;
        }
        long l = this.getMaxStale(httpRequest);
        if (l == -1L) {
            return false;
        }
        return l > this.validityStrategy.getStalenessSecs(httpCacheEntry, date);
    }

    private boolean originInsistsOnFreshness(HttpCacheEntry httpCacheEntry) {
        if (this.validityStrategy.mustRevalidate(httpCacheEntry)) {
            return true;
        }
        if (!this.sharedCache) {
            return false;
        }
        return this.validityStrategy.proxyRevalidate(httpCacheEntry) || this.validityStrategy.hasCacheControlDirective(httpCacheEntry, "s-maxage");
    }

    private long getMaxStale(HttpRequest httpRequest) {
        long l = -1L;
        for (Header header : httpRequest.getHeaders("Cache-Control")) {
            for (HeaderElement headerElement : header.getElements()) {
                if (!"max-stale".equals(headerElement.getName())) continue;
                if ((headerElement.getValue() == null || "".equals(headerElement.getValue().trim())) && l == -1L) {
                    l = Long.MAX_VALUE;
                    continue;
                }
                try {
                    long l2 = Long.parseLong(headerElement.getValue());
                    if (l2 < 0L) {
                        l2 = 0L;
                    }
                    if (l != -1L && l2 >= l) continue;
                    l = l2;
                }
                catch (NumberFormatException numberFormatException) {
                    l = 0L;
                }
            }
        }
        return l;
    }

    public boolean canCachedResponseBeUsed(HttpHost httpHost, HttpRequest httpRequest, HttpCacheEntry httpCacheEntry, Date date) {
        if (!this.isFreshEnough(httpCacheEntry, httpRequest, date)) {
            this.log.trace("Cache entry was not fresh enough");
            return false;
        }
        if (this.isGet(httpRequest) && !this.validityStrategy.contentLengthHeaderMatchesActualLength(httpCacheEntry)) {
            this.log.debug("Cache entry Content-Length and header information do not match");
            return false;
        }
        if (this.hasUnsupportedConditionalHeaders(httpRequest)) {
            this.log.debug("Request contained conditional headers we don't handle");
            return false;
        }
        if (!this.isConditional(httpRequest) && httpCacheEntry.getStatusCode() == 304) {
            return false;
        }
        if (this.isConditional(httpRequest) && !this.allConditionalsMatch(httpRequest, httpCacheEntry, date)) {
            return false;
        }
        if (this.hasUnsupportedCacheEntryForGet(httpRequest, httpCacheEntry)) {
            this.log.debug("HEAD response caching enabled but the cache entry does not contain a request method, entity or a 204 response");
            return false;
        }
        for (Header header : httpRequest.getHeaders("Cache-Control")) {
            for (HeaderElement headerElement : header.getElements()) {
                int n;
                if ("no-cache".equals(headerElement.getName())) {
                    this.log.trace("Response contained NO CACHE directive, cache was not suitable");
                    return false;
                }
                if ("no-store".equals(headerElement.getName())) {
                    this.log.trace("Response contained NO STORE directive, cache was not suitable");
                    return false;
                }
                if ("max-age".equals(headerElement.getName())) {
                    try {
                        n = Integer.parseInt(headerElement.getValue());
                        if (this.validityStrategy.getCurrentAgeSecs(httpCacheEntry, date) > (long)n) {
                            this.log.trace("Response from cache was NOT suitable due to max age");
                            return false;
                        }
                    }
                    catch (NumberFormatException numberFormatException) {
                        this.log.debug("Response from cache was malformed" + numberFormatException.getMessage());
                        return false;
                    }
                }
                if ("max-stale".equals(headerElement.getName())) {
                    try {
                        n = Integer.parseInt(headerElement.getValue());
                        if (this.validityStrategy.getFreshnessLifetimeSecs(httpCacheEntry) > (long)n) {
                            this.log.trace("Response from cache was not suitable due to Max stale freshness");
                            return false;
                        }
                    }
                    catch (NumberFormatException numberFormatException) {
                        this.log.debug("Response from cache was malformed: " + numberFormatException.getMessage());
                        return false;
                    }
                }
                if (!"min-fresh".equals(headerElement.getName())) continue;
                try {
                    long l = Long.parseLong(headerElement.getValue());
                    if (l < 0L) {
                        return false;
                    }
                    long l2 = this.validityStrategy.getCurrentAgeSecs(httpCacheEntry, date);
                    long l3 = this.validityStrategy.getFreshnessLifetimeSecs(httpCacheEntry);
                    if (l3 - l2 >= l) continue;
                    this.log.trace("Response from cache was not suitable due to min fresh freshness requirement");
                    return false;
                }
                catch (NumberFormatException numberFormatException) {
                    this.log.debug("Response from cache was malformed: " + numberFormatException.getMessage());
                    return false;
                }
            }
        }
        this.log.trace("Response from cache was suitable");
        return true;
    }

    private boolean isGet(HttpRequest httpRequest) {
        return httpRequest.getRequestLine().getMethod().equals("GET");
    }

    private boolean entryIsNotA204Response(HttpCacheEntry httpCacheEntry) {
        return httpCacheEntry.getStatusCode() != 204;
    }

    private boolean cacheEntryDoesNotContainMethodAndEntity(HttpCacheEntry httpCacheEntry) {
        return httpCacheEntry.getRequestMethod() == null && httpCacheEntry.getResource() == null;
    }

    private boolean hasUnsupportedCacheEntryForGet(HttpRequest httpRequest, HttpCacheEntry httpCacheEntry) {
        return this.isGet(httpRequest) && this.cacheEntryDoesNotContainMethodAndEntity(httpCacheEntry) && this.entryIsNotA204Response(httpCacheEntry);
    }

    public boolean isConditional(HttpRequest httpRequest) {
        return this.hasSupportedEtagValidator(httpRequest) || this.hasSupportedLastModifiedValidator(httpRequest);
    }

    public boolean allConditionalsMatch(HttpRequest httpRequest, HttpCacheEntry httpCacheEntry, Date date) {
        boolean bl;
        boolean bl2 = this.hasSupportedEtagValidator(httpRequest);
        boolean bl3 = this.hasSupportedLastModifiedValidator(httpRequest);
        boolean bl4 = bl2 && this.etagValidatorMatches(httpRequest, httpCacheEntry);
        boolean bl5 = bl = bl3 && this.lastModifiedValidatorMatches(httpRequest, httpCacheEntry, date);
        if (bl2 && bl3 && (!bl4 || !bl)) {
            return false;
        }
        if (bl2 && !bl4) {
            return false;
        }
        return !bl3 || bl;
    }

    private boolean hasUnsupportedConditionalHeaders(HttpRequest httpRequest) {
        return httpRequest.getFirstHeader("If-Range") != null || httpRequest.getFirstHeader("If-Match") != null || this.hasValidDateField(httpRequest, "If-Unmodified-Since");
    }

    private boolean hasSupportedEtagValidator(HttpRequest httpRequest) {
        return httpRequest.containsHeader("If-None-Match");
    }

    private boolean hasSupportedLastModifiedValidator(HttpRequest httpRequest) {
        return this.hasValidDateField(httpRequest, "If-Modified-Since");
    }

    private boolean etagValidatorMatches(HttpRequest httpRequest, HttpCacheEntry httpCacheEntry) {
        Header header = httpCacheEntry.getFirstHeader("ETag");
        String string = header != null ? header.getValue() : null;
        Header[] headerArray = httpRequest.getHeaders("If-None-Match");
        if (headerArray != null) {
            for (Header header2 : headerArray) {
                for (HeaderElement headerElement : header2.getElements()) {
                    String string2 = headerElement.toString();
                    if ((!"*".equals(string2) || string == null) && !string2.equals(string)) continue;
                    return true;
                }
            }
        }
        return false;
    }

    private boolean lastModifiedValidatorMatches(HttpRequest httpRequest, HttpCacheEntry httpCacheEntry, Date date) {
        Header header = httpCacheEntry.getFirstHeader("Last-Modified");
        Date date2 = null;
        if (header != null) {
            date2 = DateUtils.parseDate(header.getValue());
        }
        if (date2 == null) {
            return false;
        }
        for (Header header2 : httpRequest.getHeaders("If-Modified-Since")) {
            Date date3 = DateUtils.parseDate(header2.getValue());
            if (date3 == null || !date3.after(date) && !date2.after(date3)) continue;
            return false;
        }
        return true;
    }

    private boolean hasValidDateField(HttpRequest httpRequest, String string) {
        int n = 0;
        Header[] headerArray = httpRequest.getHeaders(string);
        int n2 = headerArray.length;
        if (n < n2) {
            Header header = headerArray[n];
            Date date = DateUtils.parseDate(header.getValue());
            return date != null;
        }
        return false;
    }
}

